<?php
/**
 * Plugin Name: Lean Bunker Native Editor
 * Description: Pure native WYSIWYG editor for ClassicPress. Stable, lightweight, with developer-friendly API.
 * Version: 0.0.11.2
 * Author: Lean Bunker
 * License: GPL-2.0+
 * Note: Built for autonomy, stability, and control. Includes LBEditor public API.
 */
if (!defined('ABSPATH')) exit;

if (!class_exists('LB_NativeEditor')) {
class LB_NativeEditor {
    const V = '0.0.11.2';
    const SLUG = 'lb-native-editor';

    private static $i = null;
    private function __construct() { $this->init(); }
    public static function get() { return self::$i ?: self::$i = new self; }

    private function init() {
        add_filter('user_can_richedit', '__return_false', 9999);
        add_action('admin_init', [$this, 'rm_tinymce'], 1);
        add_filter('the_editor', [$this, 'replace_editor'], 10, 1);
        add_action('admin_head', [$this, 'add_styles'], 9999);
        add_action('admin_footer', [$this, 'add_editor_script'], 9999);
        add_filter('wp_kses_allowed_html', [$this, 'allow_lb_attributes'], 10, 2);
    }

    public function rm_tinymce() {
        remove_action('admin_print_footer_scripts', 'wp_tiny_mce', 50);
        remove_action('admin_head', 'wp_enqueue_editor', 10);
        foreach (get_post_types(['public' => true], 'names') as $t) {
            remove_post_type_support($t, 'editor');
            add_post_type_support($t, 'editor');
        }
    }

    public function replace_editor($output) {
        global $pagenow, $post;
        if (!in_array($pagenow, ['post.php', 'post-new.php'], true)) return $output;

        $content = '<p>Write here...</p>';
        $raw_content = '';
        if ($post && !empty($post->ID)) {
            $raw_content = get_post_field('post_content', $post->ID, 'raw');
            if (!empty(trim($raw_content))) {
                $content = $raw_content;
            }
        }

        return '
        <div id="wp-content-wrap">
            <div id="lb-editor-toolbar">
                <select id="lb-heading-select" title="Text formatting" style="padding:5px 22px 5px 8px;font-size:12px;border:1px solid #ddd;border-radius:3px;background:#fff;margin-right:8px;">
                    <option value="">Normal</option>
                    <option value="h1">Heading 1</option>
                    <option value="h2">Heading 2</option>
                    <option value="h3">Heading 3</option>
                    <option value="h4">Heading 4</option>
                    <option value="p">Paragraph</option>
                    <option value="pre">Code</option>
                </select>
                <span class="lb-btn" data-cmd="bold" title="Bold (Ctrl+B)">B</span>
                <span class="lb-btn" data-cmd="italic" title="Italic (Ctrl+I)">I</span>
                <span class="lb-btn" data-cmd="underline" title="Underline (Ctrl+U)">U</span>
                <span class="lb-btn" data-cmd="strikeThrough" title="Strikethrough">S</span>
                <span class="lb-btn" data-cmd="justifyLeft" title="Align left">⇤</span>
                <span class="lb-btn" data-cmd="justifyCenter" title="Align center">☰</span>
                <span class="lb-btn" data-cmd="justifyRight" title="Align right">⇥</span>
                <span class="lb-btn" data-cmd="justifyFull" title="Justify">≡</span>
                <span class="lb-btn" data-cmd="insertUnorderedList" title="Bullet list">•</span>
                <span class="lb-btn" data-cmd="insertOrderedList" title="Numbered list">1.</span>
                <span class="lb-btn" data-cmd="blockquote" title="Quote">❝</span>
                <span class="lb-btn" data-cmd="createLink" title="Insert link">🔗</span>
                <span class="lb-btn" data-cmd="unlink" title="Remove link">×</span>
                <span class="lb-btn" id="lb-media-btn" title="Insert media">🖼️</span>
                <span class="lb-btn" data-cmd="removeFormat" title="Remove formatting">⌫</span>
            </div>
            <div id="wp-content-editor-container">
                <div id="lb-content-editable" contenteditable="true" spellcheck="true" class="lb-wysiwyg-editor">' .
                    $content .
                '</div>
                <textarea id="content" name="content" style="display:none;">' .
                    esc_textarea($raw_content) .
                '</textarea>
            </div>
        </div>';
    }

    public function allow_lb_attributes($tags, $context) {
        if ($context === 'post') {
            $attrs = ['data-lb-fix-id', 'data-lb-strategy', 'data-lb-experiment-id', 'data-lb-group'];
            foreach (['a', 'span', 'p', 'div', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'blockquote'] as $tag) {
                if (!isset($tags[$tag])) $tags[$tag] = [];
                foreach ($attrs as $attr) {
                    $tags[$tag][$attr] = true;
                }
            }
        }
        return $tags;
    }

    public function add_styles() {
        global $pagenow;
        if (!in_array($pagenow, ['post.php', 'post-new.php'], true)) return;
        echo '<style id="lb-native-editor-styles">
/* Hide legacy editors */
#content-tmce, #content-html, .wp-switch-editor, #wp-content-editor-tools .wp-editor-tabs,
.quicktags-toolbar, #mce_fullscreen_container, .mce-container, .ed_toolbar,
.wp-editor-expand .wp-editor-tools { display:none !important; visibility:hidden !important; }
/* Toolbar */
#lb-editor-toolbar {
    background:#f8f9fa; border:1px solid #ddd; border-radius:4px 4px 0 0;
    padding:8px 12px; display:flex; gap:6px; align-items:center;
    font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,Oxygen-Sans,Ubuntu,Cantarell,"Helvetica Neue",sans-serif;
    box-shadow:0 1px 1px rgba(0,0,0,0.04); z-index:9999;
}
#lb-heading-select { cursor:pointer; appearance:auto; }
.lb-btn {
    width:28px; height:26px; display:flex; align-items:center; justify-content:center;
    background:#fff; border:1px solid #ccc; border-radius:3px; font-size:15px; font-weight:500;
    color:#333; user-select:none; cursor:pointer; transition:all 0.15s;
}
.lb-btn:hover { background:#e9f0f7; border-color:#007cba; color:#007cba; }
.lb-btn:active { background:#007cba; color:#fff; }
/* Editor */
.lb-wysiwyg-editor {
    padding:20px; border:2px solid #ddd; border-top:none; border-radius:0 0 4px 4px;
    background:#fff; min-height:500px; width:100%; box-sizing:border-box;
    outline:none; resize:vertical; overflow-y:auto; word-wrap:break-word;
    box-shadow:inset 0 1px 2px rgba(0,0,0,0.07);
    font-family:inherit !important; font-size:inherit !important;
    line-height:inherit !important; color:inherit !important;
}
.lb-wysiwyg-editor:focus {
    border-color:#007cba;
    box-shadow:0 0 0 3px rgba(0,123,255,0.15), inset 0 1px 2px rgba(0,0,0,0.07);
}
/* Structural only */
.lb-wysiwyg-editor img { max-width:100% !important; height:auto !important; display:block !important; margin:1em auto !important; }
.lb-wysiwyg-editor ul { list-style-type:disc !important; padding-left:35px !important; margin:1em 0 !important; }
.lb-wysiwyg-editor ol { list-style-type:decimal !important; padding-left:35px !important; margin:1em 0 !important; }
.lb-wysiwyg-editor blockquote {
    border-left:3px solid #ddd !important; padding:0.8em 1.2em !important;
    margin:1.2em 0 !important; background:rgba(0,0,0,0.02) !important;
}
.lb-wysiwyg-editor a { color:#007cba !important; text-decoration:underline !important; }
.lb-wysiwyg-editor pre,
.lb-wysiwyg-editor code {
    background:#f8f9fa !important; padding:0.2em 0.4em !important;
    border-radius:3px !important; font-family:monospace !important;
}
.lb-wysiwyg-editor pre {
    display:block !important; padding:1em !important; margin:1em 0 !important;
    overflow-x:auto !important;
}
</style>';
    }

    public function add_editor_script() {
        global $pagenow;
        if (!in_array($pagenow, ['post.php', 'post-new.php'], true)) return;
        echo '<script>
window.addEventListener("pageshow", function(event) {
    const editor = document.getElementById("lb-content-editable");
    const textarea = document.getElementById("content");
    if (!editor || !textarea) return;

    let syncTimer;
    let isSyncing = false;

    function syncTextarea() {
        if (isSyncing) return;
        isSyncing = true;
        try {
            let html = editor.innerHTML;
            if (html === "<p>Write here...</p>") html = "";
            textarea.value = html;
        } finally {
            isSyncing = false;
        }
    }

    function scheduleSync() {
        clearTimeout(syncTimer);
        syncTimer = setTimeout(syncTextarea, 800);
    }

    // Reconcile on load
    const initialTextarea = textarea.value;
    const initialEditor = editor.innerHTML;
    const isPlaceholder = initialEditor === "<p>Write here...</p>";
    const normalizedEditor = isPlaceholder ? "" : initialEditor.replace(/\\s+/g, "");
    const normalizedTextarea = initialTextarea.replace(/\\s+/g, "");

    if (!isPlaceholder && normalizedEditor !== normalizedTextarea) {
        textarea.value = initialEditor;
    }

    syncTextarea();

    // User edits
    editor.addEventListener("input", () => scheduleSync());
    editor.addEventListener("paste", () => scheduleSync());
    editor.addEventListener("keydown", () => scheduleSync());

    // Non-user edits
    const mutationObserver = new MutationObserver(scheduleSync);
    mutationObserver.observe(editor, {
        childList: true,
        subtree: true,
        attributes: true,
        attributeFilter: ["data-lb-fix-id", "data-lb-strategy", "class", "style", "href"]
    });

    // Placeholder
    editor.addEventListener("focus", () => {
        if (editor.innerHTML === "<p>Write here...</p>") editor.innerHTML = "";
    });
    editor.addEventListener("blur", () => {
        if (editor.innerHTML.trim() === "") editor.innerHTML = "<p>Write here...</p>";
    });

    // Toolbar
    function execCmd(command, value) {
        editor.focus();
        try {
            document.execCommand(command, false, value);
            scheduleSync();
        } catch(e) {
            console.warn("execCommand failed:", e);
        }
    }

    const headingSelect = document.getElementById("lb-heading-select");
    if (headingSelect) {
        headingSelect.addEventListener("change", function() {
            const tag = this.value;
            if (!tag) return;
            editor.focus();
            const selection = window.getSelection();
            if (selection.rangeCount === 0) return;
            let blockElement = selection.anchorNode;
            while (blockElement && blockElement.nodeType !== 1) {
                blockElement = blockElement.parentNode;
            }
            while (blockElement && !["P", "H1", "H2", "H3", "H4", "H5", "H6", "PRE", "BLOCKQUOTE"].includes(blockElement.tagName)) {
                blockElement = blockElement.parentNode;
            }
            if (blockElement) {
                const newElement = document.createElement(tag);
                newElement.innerHTML = blockElement.innerHTML;
                blockElement.parentNode.replaceChild(newElement, blockElement);
                const newRange = document.createRange();
                newRange.selectNodeContents(newElement);
                newRange.collapse(false);
                selection.removeAllRanges();
                selection.addRange(newRange);
                scheduleSync();
            } else {
                execCmd("formatBlock", "<" + tag + ">");
            }
            this.value = "";
        });
    }

    document.querySelectorAll("#lb-editor-toolbar .lb-btn").forEach(btn => {
        btn.addEventListener("click", function(e) {
            e.preventDefault();
            const cmd = this.dataset.cmd;
            if (cmd === "createLink") {
                const url = prompt("Enter URL:", "https://");
                if (url && url.trim()) {
                    execCmd("createLink", url.trim());
                    const sel = window.getSelection();
                    if (sel.anchorNode) {
                        let linkNode = sel.anchorNode.parentNode;
                        while (linkNode && linkNode.tagName !== "A") linkNode = linkNode.parentNode;
                        if (linkNode && linkNode.tagName === "A") {
                            linkNode.removeAttribute("target");
                            linkNode.removeAttribute("rel");
                        }
                    }
                }
            } else if (cmd === "unlink") {
                execCmd("unlink", null);
            } else if (cmd === "blockquote") {
                editor.focus();
                const selection = window.getSelection();
                if (selection.rangeCount === 0) return;
                let node = selection.anchorNode;
                while (node && node.nodeType !== 1) node = node.parentNode;
                while (node && !["P", "H1", "H2", "H3", "H4", "H5", "H6", "PRE", "BLOCKQUOTE"].includes(node.tagName)) {
                    node = node.parentNode;
                }
                if (node && node.tagName === "BLOCKQUOTE") {
                    const newElement = document.createElement("p");
                    newElement.innerHTML = node.innerHTML;
                    node.parentNode.replaceChild(newElement, node);
                    const newRange = document.createRange();
                    newRange.selectNodeContents(newElement);
                    newRange.collapse(false);
                    selection.removeAllRanges();
                    selection.addRange(newRange);
                    scheduleSync();
                } else {
                    execCmd("formatBlock", "<blockquote>");
                }
            } else if (cmd === "insertUnorderedList" || cmd === "insertOrderedList") {
                execCmd(cmd, null);
            } else if (["justifyLeft", "justifyCenter", "justifyRight", "justifyFull"].includes(cmd)) {
                execCmd(cmd, null);
            } else {
                execCmd(cmd, null);
            }
        });
    });

    // Media
    const mediaBtn = document.getElementById("lb-media-btn");
    if (mediaBtn && typeof wp !== "undefined" && wp.media) {
        mediaBtn.addEventListener("click", function(e) {
            e.preventDefault();
            const frame = wp.media({ title: "Insert Media", button: { text: "Insert" }, multiple: false });
            frame.on("select", function() {
                const attachment = frame.state().get("selection").first().toJSON();
                editor.focus();
                if (attachment.type === "image") {
                    try {
                        document.execCommand("insertImage", false, attachment.url);
                    } catch (err) {
                        document.execCommand("insertHTML", false, "<img src=\\"" + attachment.url + "\\" alt=\\"" + (attachment.alt || "") + "\\" />");
                    }
                } else if (attachment.type === "video") {
                    document.execCommand("insertHTML", false, "<video src=\\"" + attachment.url + "\\" controls></video>");
                } else if (attachment.type === "audio") {
                    document.execCommand("insertHTML", false, "<audio src=\\"" + attachment.url + "\\" controls></audio>");
                } else {
                    document.execCommand("insertHTML", false, "<a href=\\"" + attachment.url + "\\">" + (attachment.title || attachment.filename) + "</a>");
                }
                scheduleSync();
            });
            frame.open();
        });
    }

    // Context menu for links
    editor.addEventListener("contextmenu", function(e) {
        if (e.target.tagName === "A") {
            e.preventDefault();
            const linkUrl = e.target.href;
            const action = confirm("Link: " + linkUrl + "\\n\\nOK = edit, Cancel = remove");
            if (action) {
                const newUrl = prompt("Edit URL:", linkUrl);
                if (newUrl && newUrl.trim()) {
                    e.target.href = newUrl.trim();
                    scheduleSync();
                }
            } else {
                const textNode = document.createTextNode(e.target.textContent);
                e.target.parentNode.replaceChild(textNode, e.target);
                scheduleSync();
            }
        }
    });

    // Sync on form submit
    const form = document.getElementById("post");
    if (form) {
        form.addEventListener("submit", syncTextarea);
    }

    // Autosave hook
    if (typeof wp !== "undefined" && wp.autosave && wp.autosave.server) {
        wp.autosave.server.addBeforeSaveCallback(function() {
            syncTextarea();
            return true;
        });
    }

    // ┌──────────────────────────────────────────────────────┐
    // │ LB Editor Public API — for developers who ❤️ control │
    // └──────────────────────────────────────────────────────┘
    window.LBEditor = window.LBEditor || {
        insertHTML: function(html) {
            if (!html || typeof html !== "string") return false;
            const editor = document.getElementById("lb-content-editable");
            if (!editor) return false;
            editor.focus();
            try {
                if (document.getSelection().rangeCount > 0) {
                    document.execCommand("insertHTML", false, html);
                } else {
                    editor.innerHTML = editor.innerHTML.replace(/(<p>Write here...<\/p>)?$/, html + "<p><br></p>");
                }
                if (typeof scheduleSync === "function") scheduleSync();
                return true;
            } catch (e) {
                console.warn("LBEditor.insertHTML failed:", e);
                return false;
            }
        },
        setContent: function(html) {
            const editor = document.getElementById("lb-content-editable");
            if (!editor) return false;
            editor.innerHTML = (html && html.trim()) ? html : "<p>Write here...</p>";
            if (typeof scheduleSync === "function") scheduleSync();
            return true;
        },
        getContent: function() {
            const editor = document.getElementById("lb-content-editable");
            if (!editor) return null;
            let html = editor.innerHTML;
            if (html === "<p>Write here...</p>") return "";
            return html;
        },
        isAvailable: function() {
            return !!document.getElementById("lb-content-editable");
        }
    };
});
</script>';
    }

    public static function instance() { return self::get(); }
}

function lb_native_editor_init() {
    if (is_admin()) LB_NativeEditor::instance();
}
add_action('plugins_loaded', 'lb_native_editor_init');
} // End class_exists

// 🔑 Universal API availability declaration
if (is_admin()) {
    define('LBEDITOR_API_AVAILABLE', true);
}